import javax.media.j3d.*;
import javax.vecmath.*; 
import java.awt.*;
import java.awt.event.*;
/**
 * This uses a Morph object to animate a shape between two key shapes:
 * a cube and a pyramid. Buttons are used to change the weight values.
 * @author I.J.Palmer
 * @version 1.0
 */
public class SimpleMorph extends Frame implements ActionListener {
	protected Canvas3D myCanvas3D = new Canvas3D(null);
	/** The button that will animate towards the cube key shape */
	protected Button cubeButton = new Button("Cube");
	/** The button that will animate towards the pyramid key shape */
	protected Button pyraButton = new Button("Pyramid");
	/** The exit button */
	protected Button exitButton = new Button("Exit");
	/** This performs the animation */
	protected Morph myMorph;
	/** The weight values for the morph */
	protected double[] weights = {0.5,0.5};

      /** 
       * Build the view branch of the scene graph 
       * @return BranchGroup that is the root of the view branch
       */
	protected BranchGroup buildViewBranch(Canvas3D c) {
		BranchGroup viewBranch = new BranchGroup();
		Transform3D viewXfm = new Transform3D();
            viewXfm.set(new Vector3f(0.0f,0.0f,5.0f));
		TransformGroup viewXfmGroup = new TransformGroup(viewXfm);
		ViewPlatform myViewPlatform = new ViewPlatform();
            PhysicalBody myBody = new PhysicalBody();
            PhysicalEnvironment myEnvironment = new PhysicalEnvironment();
		viewXfmGroup.addChild(myViewPlatform);
		viewBranch.addChild(viewXfmGroup);
		View myView = new View();
		myView.addCanvas3D(c);
		myView.attachViewPlatform(myViewPlatform);
            myView.setPhysicalBody(myBody);
            myView.setPhysicalEnvironment(myEnvironment);
		return viewBranch;
	}

      /** 
       * Add some lights to the scene graph 
       * @param b BranchGroup that the lights are added to
       */
      protected void addLights(BranchGroup b) {
            BoundingSphere bounds = new BoundingSphere(new Point3d(0.0,0.0,0.0), 100.0);
		Color3f ambLightColour = new Color3f(0.5f, 0.5f, 0.5f);
		AmbientLight ambLight = new AmbientLight(ambLightColour);
            ambLight.setInfluencingBounds(bounds);
            Color3f dirLightColour = new Color3f(1.0f, 1.0f, 1.0f);
            Vector3f dirLightDir  = new Vector3f(-1.0f, -1.0f, -1.0f);
            DirectionalLight dirLight = new DirectionalLight(dirLightColour, dirLightDir);
            dirLight.setInfluencingBounds(bounds);
		b.addChild(ambLight);
            b.addChild(dirLight);
      }

      /** 
       * Create the Morph from the given shapes 
       * @param theShapes GeometryArray that stores the shapes for the Morph
       * @param app Appearnce used for the shapes
       * @return Morph that uses the given shapes
       */
	protected Morph createMorph(GeometryArray[] theShapes, Appearance app) {
            //Create the morph from the given shapes and appearance
		myMorph = new Morph(theShapes,app);
		//Set the weights
		myMorph.setWeights(weights);
		//Set the capabilities so that we can read and write the weights
		myMorph.setCapability(Morph.ALLOW_WEIGHTS_READ);
		myMorph.setCapability(Morph.ALLOW_WEIGHTS_WRITE);
		
		return myMorph;
	}

      /** 
       * Build the content branch for the scene graph
       * @return BranchGroup that is the root of the content
       */
      protected BranchGroup buildContentBranch() {
            //Create the appearance object
		Appearance app = new Appearance();
            Color3f ambientColour = new Color3f(1.0f,0.0f,0.0f);
            Color3f emissiveColour = new Color3f(0.0f,0.0f,0.0f);
		Color3f specularColour = new Color3f(1.0f,1.0f,1.0f);
            Color3f diffuseColour = new Color3f(1.0f,0.0f,0.0f);
            float shininess = 20.0f;
            app.setMaterial(new Material(ambientColour,emissiveColour,
                            diffuseColour,specularColour,shininess));
		//Make the cube key shape
  		IndexedQuadArray indexedCube = new IndexedQuadArray(8,
				IndexedQuadArray.COORDINATES|IndexedQuadArray.NORMALS, 24);
		Point3f[] cubeCoordinates = { new Point3f( 1.0f, 1.0f, 1.0f),
					            new Point3f(-1.0f, 1.0f, 1.0f),
					            new Point3f(-1.0f,-1.0f, 1.0f),
					            new Point3f( 1.0f,-1.0f, 1.0f),
					            new Point3f( 1.0f, 1.0f,-1.0f),
					            new Point3f(-1.0f, 1.0f,-1.0f),
					            new Point3f(-1.0f,-1.0f,-1.0f),
					            new Point3f( 1.0f,-1.0f,-1.0f)};
		Vector3f[] cubeNormals= {  new Vector3f( 0.0f, 0.0f, 1.0f),
                                       new Vector3f( 0.0f, 0.0f,-1.0f),
                                       new Vector3f( 1.0f, 0.0f, 0.0f),
                                       new Vector3f(-1.0f, 0.0f, 0.0f),
                                       new Vector3f( 0.0f, 1.0f, 0.0f),
                                       new Vector3f( 0.0f,-1.0f, 0.0f)};
		int cubeCoordIndices[] = {0,1,2,3,7,6,5,4,0,3,7,4,5,6,2,1,0,4,5,1,6,7,3,2};
		int cubeNormalIndices[] = {0,0,0,0,1,1,1,1,2,2,2,2,3,3,3,3,4,4,4,4,5,5,5,5};
		indexedCube.setCoordinates(0, cubeCoordinates);
		indexedCube.setNormals(0,cubeNormals);
		indexedCube.setCoordinateIndices(0, cubeCoordIndices);
		indexedCube.setNormalIndices(0, cubeNormalIndices);
		//Make the pyramid key shape.  Although this needs
		//only five vertices to create the desired shape, we 
		//need to use six vertices so that it has the same 
		//number as the cube.
 		IndexedQuadArray indexedPyramid = new IndexedQuadArray(8,
				IndexedQuadArray.COORDINATES|IndexedQuadArray.NORMALS, 24);
		Point3f[] pyramidCoordinates = {  new Point3f( 0.0f, 1.0f, 0.0f),
					                new Point3f(0.0f, 1.0f, 0.0f),
    					                new Point3f(-1.0f,-1.0f, 1.0f),
					                new Point3f( 1.0f,-1.0f, 1.0f),
					                new Point3f( 0.0f, 1.0f,0.0f),
					                new Point3f(0.0f, 1.0f,0.0f),
					                new Point3f(-1.0f,-1.0f,-1.0f),
					                new Point3f( 1.0f,-1.0f,-1.0f)};
		Vector3f[] pyramidNormals= {   new Vector3f( 0.0f, 0.0f, 1.0f),
                                           new Vector3f( 0.0f, 0.0f,-1.0f),
                                           new Vector3f( 1.0f, 0.0f, 0.0f),
                                           new Vector3f(-1.0f, 0.0f, 0.0f),
                                           new Vector3f( 0.0f, 1.0f, 0.0f),
                                           new Vector3f( 0.0f,-1.0f, 0.0f)};
		int pyramidCoordIndices[] = {0,1,2,3,7,6,5,4,0,3,7,4,5,6,2,1,0,4,5,1,6,7,3,2};
		int pyramidNormalIndices[] = {0,0,0,0,1,1,1,1,2,2,2,2,3,3,3,3,4,4,4,4,5,5,5,5};
		indexedPyramid.setCoordinates(0, pyramidCoordinates);
		indexedPyramid.setNormals(0,pyramidNormals);
		indexedPyramid.setCoordinateIndices(0, pyramidCoordIndices);
		indexedPyramid.setNormalIndices(0, pyramidNormalIndices);
            //Set the contents of the array to the two shapes
		GeometryArray[] theShapes = new GeometryArray[2];
		theShapes[0] = indexedCube;
		theShapes[1] = indexedPyramid;
		BranchGroup contentBranch = new BranchGroup();
		//Create a transform to rotate the shape slightly
		Transform3D rotateCube = new Transform3D( );
		rotateCube.set(new AxisAngle4d(1.0,1.0,0.0,Math.PI/4.0));
		TransformGroup rotationGroup = new TransformGroup(rotateCube);
		contentBranch.addChild(rotationGroup);
            addLights(contentBranch);
            //Call the function to build the morph
		rotationGroup.addChild(createMorph(theShapes,app));
		return contentBranch;
      } 
      
      /** 
       * Process the actions from the buttons.
       * The two shape buttons change the geometry so that it 
       * is nearer to its corresponding key shape.
       * @param e ActionEvent to be processed
       */
	public void actionPerformed(ActionEvent e) {
	      //If its the exit button, quit the program
		if (e.getSource() == exitButton) {
			dispose();
                	System.exit(0);
		}
		//If its the cube button, increase the weight
		//associated with the cube key shape and reduce 
		//the other.
		else if (e.getSource() == cubeButton) {
			if (weights[0] <= 0.9) {
				weights[0] += 0.1;
				weights[1] -= 0.1;
				myMorph.setWeights(weights);
			}
		}
		//If its the pyramid button, increase the weight
		//associated with the pyramid key shape and reduce
		//the cube weight
		else if (e.getSource() == pyraButton) {
			if (weights[1] <= 0.9) {
				weights[0] -= 0.1;
				weights[1] += 0.1;
				myMorph.setWeights(weights);
			}
		}
	}
	
	/** 
	 * Constructor that generates the content using the class methods
	 */
      public SimpleMorph() {
		VirtualUniverse myUniverse = new VirtualUniverse();
		Locale myLocale = new Locale(myUniverse);
            myLocale.addBranchGraph(buildViewBranch(myCanvas3D));
            myLocale.addBranchGraph(buildContentBranch());
		setTitle("SimpleMorph");
		setSize(400,400);
		setLayout(new BorderLayout());
		Panel bottom = new Panel();
		bottom.add(pyraButton);
		bottom.add(cubeButton);
		bottom.add(exitButton);
            add(BorderLayout.CENTER, myCanvas3D);
            add(BorderLayout.SOUTH, bottom);
		pyraButton.addActionListener(this);
		cubeButton.addActionListener(this);
            exitButton.addActionListener(this);
            setVisible(true);
	}
	public static void main(String[] args) {
                  SimpleMorph sw = new SimpleMorph();
	}
}

